using System;
using System.Text.Json;
using System.Threading.Tasks;
using Wisej.AzureAdSample.Authentication;
using Wisej.AzureAdSample.Models;

namespace Wisej.AzureAdSample.Services
{
	/// <summary>
	/// Azure AD sign-in and basic user profile retrieval via Microsoft Graph.
	/// </summary>
	public class AzureAdService : AzureAdSSO
	{
		private const string GraphApiUrl = "https://graph.microsoft.com/v1.0";

		private AzureAdUser _cachedUser;
		private string _cachedUserAccessToken;

		public AzureAdService(string tenantId, string clientId, string clientSecret)
			: base(tenantId, clientId, clientSecret)
		{
		}

		public AzureAdService(string tenantId, string clientId, string clientSecret, string scope)
			: base(tenantId, clientId, clientSecret, scope)
		{
		}

		public Task<bool> LoginAsync()
		{
			return AuthenticateAsync();
		}

		public async Task<AzureAdUser> GetCurrentUserAsync()
		{
			await EnsureAuthenticatedAsync();

			if (_cachedUser != null && string.Equals(_cachedUserAccessToken, AccessToken, StringComparison.Ordinal))
				return _cachedUser;

			var url = $"{GraphApiUrl}/me?$select=id,displayName,givenName,surname,mail,userPrincipalName";
			var response = await oauthHttpClient.GetAsync(url);
			response.EnsureSuccessStatusCode();

			var text = await response.Content.ReadAsStringAsync();
			_cachedUser = JsonSerializer.Deserialize<AzureAdUser>(text);
			_cachedUserAccessToken = AccessToken;

			return _cachedUser;
		}

		public void SignOut()
		{
			_cachedUser = null;
			_cachedUserAccessToken = null;

			AccessToken = null;
			oauthHttpClient.DefaultRequestHeaders.Authorization = null;
		}

		private async Task EnsureAuthenticatedAsync()
		{
			if (string.IsNullOrWhiteSpace(AccessToken))
				await AuthenticateAsync();
		}
	}
}

