using System;
using System.Collections.Generic;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Threading.Tasks;

namespace Wisej.AzureAdSample.Authentication
{
	/// <summary>
	/// Minimal OAuth2 Authorization Code helper for Wisej.NET apps.
	/// </summary>
	public abstract class SSOBase
	{
		protected readonly string ClientId;
		protected readonly string ClientSecret;
		protected readonly HttpClient oauthHttpClient;
		public string AccessToken;

		private TaskCompletionSource<string> _authCompletionSource;
		private string _oauthState;

		protected SSOBase(string clientId, string clientSecret)
		{
			ClientId = clientId;
			ClientSecret = clientSecret;

			oauthHttpClient = new HttpClient();
			oauthHttpClient.Timeout = TimeSpan.FromMinutes(60);
			oauthHttpClient.DefaultRequestHeaders.UserAgent.ParseAdd("Wisej.AzureAdSample");
		}

		protected virtual string AuthorizationUrl { get; set; }
		protected virtual string TokenUrl { get; set; }
		protected virtual string Scope { get; set; }

		public virtual async Task<bool> AuthenticateAsync()
		{
			Wisej.Web.Application.ApplicationRefresh += OAuthCallbackHandler;

			_oauthState = Guid.NewGuid().ToString();
			var redirectUri = WebUtility.UrlEncode(Wisej.Web.Application.Url);
			var authUrl = BuildAuthorizationUrl(redirectUri, _oauthState, WebUtility.UrlEncode(Scope));

			_authCompletionSource = new TaskCompletionSource<string>();
			Wisej.Web.Application.Navigate(authUrl);

			AccessToken = await _authCompletionSource.Task;
			return !string.IsNullOrWhiteSpace(AccessToken);
		}

		protected virtual string BuildAuthorizationUrl(string redirectUriEncoded, string state, string scopeEncoded)
		{
			return $"{AuthorizationUrl}?client_id={ClientId}&redirect_uri={redirectUriEncoded}&state={state}&scope={scopeEncoded}";
		}

		public virtual async void OAuthCallbackHandler(object sender, EventArgs e)
		{
			Wisej.Web.Application.ApplicationRefresh -= OAuthCallbackHandler;

			var state = Wisej.Web.Application.QueryString["state"];
			if (state != _oauthState)
			{
				_authCompletionSource.TrySetException(new InvalidOperationException("Invalid OAuth state (CSRF detected)."));
				return;
			}

			var code = Wisej.Web.Application.QueryString["code"];
			if (string.IsNullOrWhiteSpace(code))
				throw new InvalidOperationException("Authorization code missing.");

			Wisej.Web.Application.Eval(
				"if (window.history && window.history.replaceState) {" +
				"  var url = new URL(window.location.href);" +
				"  url.searchParams.delete('code');" +
				"  url.searchParams.delete('state');" +
				"  history.replaceState(null, document.title, url.toString());" +
				"}");

			AccessToken = await ExchangeCodeForTokenAsync(code);
			oauthHttpClient.DefaultRequestHeaders.Authorization =
				new AuthenticationHeaderValue("Bearer", AccessToken);

			_authCompletionSource.TrySetResult(AccessToken);
		}

		public virtual async Task<string> ExchangeCodeForTokenAsync(string code)
		{
			var postData = new Dictionary<string, string>
			{
				{ "client_id", ClientId },
				{ "client_secret", ClientSecret },
				{ "code", code },
				{ "redirect_uri", Wisej.Web.Application.StartupUrl }
			};

			var response = await oauthHttpClient.PostAsync(TokenUrl, new FormUrlEncodedContent(postData));
			response.EnsureSuccessStatusCode();

			var responseContent = await response.Content.ReadAsStringAsync();
			var parsedContent = ParseQueryStringNet(responseContent);

			if (!parsedContent.TryGetValue("access_token", out var token))
				return null;

			return token;
		}

		private static Dictionary<string, string> ParseQueryStringNet(string query)
		{
			var result = new Dictionary<string, string>(StringComparer.OrdinalIgnoreCase);

			if (string.IsNullOrEmpty(query))
				return result;

			if (query.StartsWith("?", StringComparison.Ordinal))
				query = query.Substring(1);

			var pairs = query.Split('&');
			foreach (var pair in pairs)
			{
				if (string.IsNullOrEmpty(pair))
					continue;

				var kv = pair.Split(new[] { '=' }, 2);
				var key = Uri.UnescapeDataString(kv[0]);
				var value = kv.Length > 1 ? Uri.UnescapeDataString(kv[1]) : string.Empty;
				result[key] = value;
			}

			return result;
		}
	}
}

