using System;
using System.Collections.Generic;
using System.Net.Http;
using System.Text.Json;
using System.Text.Json.Serialization;
using System.Threading.Tasks;

namespace Wisej.AzureAdSample.Authentication
{
	/// <summary>
	/// Azure AD (Microsoft identity platform) OAuth2 Authorization Code flow (v2.0).
	/// </summary>
	public class AzureAdSSO : SSOBase
	{
		public const string DefaultScope = "openid profile email offline_access User.Read";

		private readonly string _tenantId;

		public AzureAdSSO(string tenantId, string clientId, string clientSecret)
			: this(tenantId, clientId, clientSecret, DefaultScope)
		{
		}

		public AzureAdSSO(string tenantId, string clientId, string clientSecret, string scope)
			: base(clientId, clientSecret)
		{
			_tenantId = string.IsNullOrWhiteSpace(tenantId) ? "common" : tenantId;
			Scope = string.IsNullOrWhiteSpace(scope) ? DefaultScope : scope;
		}

		protected override string AuthorizationUrl => $"https://login.microsoftonline.com/{_tenantId}/oauth2/v2.0/authorize";
		protected override string TokenUrl => $"https://login.microsoftonline.com/{_tenantId}/oauth2/v2.0/token";

		protected override string BuildAuthorizationUrl(string redirectUriEncoded, string state, string scopeEncoded)
		{
			return $"{AuthorizationUrl}?client_id={ClientId}&response_type=code&response_mode=query&redirect_uri={redirectUriEncoded}&state={state}&scope={scopeEncoded}";
		}

		public override async Task<string> ExchangeCodeForTokenAsync(string code)
		{
			var postData = new Dictionary<string, string>
			{
				{ "client_id", ClientId },
				{ "client_secret", ClientSecret },
				{ "grant_type", "authorization_code" },
				{ "code", code },
				{ "redirect_uri", Wisej.Web.Application.StartupUrl },
				{ "scope", Scope },
			};

			var response = await oauthHttpClient.PostAsync(TokenUrl, new FormUrlEncodedContent(postData));
			response.EnsureSuccessStatusCode();

			var responseContent = await response.Content.ReadAsStringAsync();
			var token = JsonSerializer.Deserialize<AzureAdTokenResponse>(responseContent);

			if (string.IsNullOrWhiteSpace(token?.AccessToken))
				throw new InvalidOperationException("Azure AD token response did not contain an access_token.");

			return token.AccessToken;
		}

		private sealed class AzureAdTokenResponse
		{
			[JsonPropertyName("access_token")]
			public string AccessToken { get; set; }
		}
	}
}
